// SPDX-License-Identifier: GPL-2.0 OR BSD-3-Clause
#include "vmlinux.h"
#include <bpf/bpf_helpers.h>
#include <bpf/bpf_tracing.h>
#include <bpf/bpf_core_read.h>

char LICENSE[] SEC("license") = "Dual BSD/GPL";

// XDP command definitions (from linux/netdev.h)
#define XDP_SETUP_PROG 0
#define XDP_SETUP_XSK_POOL 6

struct event {
    __u64 dev_ptr;
    __u64 bpf_ptr;
    __u32 command;
    __s32 return_value;
    __u8 is_xdp_setup_prog;
    __u8 is_xdp_setup_xsk_pool;
    __u8 is_unknown;
    __u8 is_entry; // 1 for entry, 0 for exit
};

struct {
    __uint(type, BPF_MAP_TYPE_RINGBUF);
    __uint(max_entries, 256 * 1024);
} events SEC(".maps");

SEC("fentry/stmmac_bpf")
int BPF_PROG(stmmac_bpf_entry, struct net_device *dev, struct netdev_bpf *bpf)
{
    struct event *e;
    __u32 command;

    // Reserve space in ringbuffer
    e = bpf_ringbuf_reserve(&events, sizeof(*e), 0);
    if (!e)
        return 0;

    // Read the command value
    command = BPF_CORE_READ(bpf, command);

    // Fill event data
    e->dev_ptr = (__u64)dev;
    e->bpf_ptr = (__u64)bpf;
    e->command = command;
    e->return_value = 0; // Not available at entry
    e->is_xdp_setup_prog = (command == XDP_SETUP_PROG);
    e->is_xdp_setup_xsk_pool = (command == XDP_SETUP_XSK_POOL);
    e->is_unknown = (command != XDP_SETUP_PROG && command != XDP_SETUP_XSK_POOL);
    e->is_entry = 1; // Mark as entry event

    bpf_ringbuf_submit(e, 0);
    return 0;
}

SEC("fexit/stmmac_bpf")
int BPF_PROG(stmmac_bpf_exit, struct net_device *dev, struct netdev_bpf *bpf, int ret)
{
    struct event *e;
    __u32 command;

    // Reserve space in ringbuffer
    e = bpf_ringbuf_reserve(&events, sizeof(*e), 0);
    if (!e)
        return 0;

    // Read the command value
    command = BPF_CORE_READ(bpf, command);

    // Fill event data
    e->dev_ptr = (__u64)dev;
    e->bpf_ptr = (__u64)bpf;
    e->command = command;
    e->return_value = ret; // Capture the return value
    e->is_xdp_setup_prog = (command == XDP_SETUP_PROG);
    e->is_xdp_setup_xsk_pool = (command == XDP_SETUP_XSK_POOL);
    e->is_unknown = (command != XDP_SETUP_PROG && command != XDP_SETUP_XSK_POOL);
    e->is_entry = 0; // Mark as exit event

    bpf_ringbuf_submit(e, 0);
    return 0;
}
