// SPDX-License-Identifier: GPL-2.0 OR BSD-3-Clause
/* Copyright (c) 2023 */
#include "vmlinux.h"
#include <bpf/bpf_helpers.h>
#include <bpf/bpf_tracing.h>
#include <bpf/bpf_core_read.h>
#include "stackcount.h"

struct {
    __uint(type, BPF_MAP_TYPE_HASH);
    __uint(max_entries, 10240);
    __type(key, struct key_t);
    __type(value, u64);
} counts SEC(".maps");

struct {
    __uint(type, BPF_MAP_TYPE_STACK_TRACE);
    __uint(max_entries, 1024);
    __type(key, u32);
} stack_traces SEC(".maps");

const volatile int filter_pid = 0;
const volatile int filter_cpu = -1;
const volatile bool perpid = false;
const volatile bool kernel_stack = true;
const volatile bool user_stack = true;

SEC("kprobe/trace_count")
int BPF_KPROBE(trace_count)
{
    u32 pid = bpf_get_current_pid_tgid() >> 32;
    
    if (filter_pid && pid != filter_pid)
        return 0;
        
    if (filter_cpu >= 0) {
        // Correct way to get current CPU
        int cur_cpu = bpf_get_smp_processor_id();
        if (cur_cpu != filter_cpu)
            return 0;
    }
    
    struct key_t key = {};
    
    if (perpid) {
        key.tgid = pid;
        bpf_get_current_comm(&key.name, sizeof(key.name));
    } else {
        key.tgid = 0xffffffff;
    }
    
    if (user_stack) {
        key.user_stack_id = bpf_get_stackid(ctx, &stack_traces, BPF_F_USER_STACK);
    } else {
        key.user_stack_id = -1;
    }
    
    if (kernel_stack) {
        key.kernel_stack_id = bpf_get_stackid(ctx, &stack_traces, 0);
    } else {
        key.kernel_stack_id = -1;
    }
    
    u64 *val, one = 1;
    val = bpf_map_lookup_elem(&counts, &key);
    if (val)
        (*val)++;
    else
        bpf_map_update_elem(&counts, &key, &one, BPF_NOEXIST);
    
    return 0;
}

char LICENSE[] SEC("license") = "Dual BSD/GPL";
