// SPDX-License-Identifier: GPL-2.0 OR BSD-3-Clause
#include "vmlinux.h"
#include <bpf/bpf_helpers.h>
#include <bpf/bpf_tracing.h>
#include <bpf/bpf_core_read.h>

char LICENSE[] SEC("license") = "Dual BSD/GPL";

#define POISON_POINTER_DELTA 0
#define NET_PTR_POISON          ((void *)(0x801 + POISON_POINTER_DELTA))

struct event {
    __u64 pool_ptr;
    __u64 netdev_ptr;
    __u8 is_null;
    __u8 is_poison;
    __u8 condition_result;
};

struct {
    __uint(type, BPF_MAP_TYPE_RINGBUF);
    __uint(max_entries, 256 * 1024);
} events SEC(".maps");

SEC("fentry/page_pool_release_retry")
int BPF_PROG(page_pool_release_retry_entry, struct work_struct *wq)
{
    struct delayed_work *dwq = (struct delayed_work *)wq;
    struct page_pool *pool;
    struct event *e;
    void *netdev;
    
    // Get the page_pool from the delayed_work
    pool = container_of(dwq, struct page_pool, release_dw);
    
    // Reserve space in ringbuffer
    e = bpf_ringbuf_reserve(&events, sizeof(*e), 0);
    if (!e)
        return 0;
    
    // Read the netdev pointer using the same READ_ONCE pattern
    netdev = BPF_CORE_READ(pool, slow.netdev);
    
    // Fill event data
    e->pool_ptr = (u64)pool;
    e->netdev_ptr = (u64)netdev;
    e->is_null = (netdev == NULL);
    e->is_poison = (netdev == (void *)NET_PTR_POISON);
    e->condition_result = (netdev == NULL || netdev == (void *)NET_PTR_POISON);
    
    bpf_ringbuf_submit(e, 0);
    return 0;
}
